<?php
require_once '../config.php';

if(!isLoggedIn()) {
    echo '<div class="text-center p-8"><p class="text-red-600">Please login first</p></div>';
    exit();
}

$transaction_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$user_id = $_SESSION['user_id'];

if(!$transaction_id) {
    echo '<div class="text-center p-8"><p class="text-red-600">Transaction not found</p></div>';
    exit();
}

// Get transaction details
$stmt = $conn->prepare("SELECT * FROM transactions WHERE id = ? AND user_id = ?");
$stmt->bind_param("ii", $transaction_id, $user_id);
$stmt->execute();
$result = $stmt->get_result();

if($result->num_rows == 0) {
    echo '<div class="text-center p-8"><p class="text-red-600">Transaction not found</p></div>';
    exit();
}

$transaction = $result->fetch_assoc();

// Format status color
$status_color = match($transaction['status']) {
    'completed' => 'green',
    'pending' => 'yellow',
    'failed' => 'red',
    default => 'gray'
};

// Format type icon and color
$type_config = match($transaction['type']) {
    'deposit' => ['icon' => 'fa-plus-circle', 'color' => 'green', 'label' => 'Deposit'],
    'withdrawal' => ['icon' => 'fa-minus-circle', 'color' => 'red', 'label' => 'Withdrawal'],
    'activation' => ['icon' => 'fa-bolt', 'color' => 'yellow', 'label' => 'Account Activation'],
    'referral_bonus' => ['icon' => 'fa-users', 'color' => 'blue', 'label' => 'Referral Bonus'],
    'product_purchase' => ['icon' => 'fa-shopping-cart', 'color' => 'purple', 'label' => 'Product Purchase'],
    default => ['icon' => 'fa-exchange-alt', 'color' => 'gray', 'label' => 'Transaction']
};

$is_positive = in_array($transaction['type'], ['deposit', 'referral_bonus']);
?>

<div class="space-y-6">
    <!-- Transaction Header -->
    <div class="text-center">
        <div class="w-20 h-20 bg-<?php echo $type_config['color']; ?>-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <i class="fas <?php echo $type_config['icon']; ?> text-<?php echo $type_config['color']; ?>-600 text-3xl"></i>
        </div>
        <h3 class="text-2xl font-bold text-gray-800"><?php echo $type_config['label']; ?></h3>
        <p class="text-gray-600">Transaction #<?php echo str_pad($transaction['id'], 6, '0', STR_PAD_LEFT); ?></p>
    </div>
    
    <!-- Amount Card -->
    <div class="bg-gradient-to-r from-<?php echo $is_positive ? 'green' : 'red'; ?>-500 to-<?php echo $is_positive ? 'emerald' : 'pink'; ?>-600 rounded-xl p-6 text-white text-center">
        <p class="text-white/80 mb-2">Transaction Amount</p>
        <div class="text-4xl font-bold">
            <?php echo $is_positive ? '+' : '-'; ?>৳<?php echo number_format($transaction['amount'], 2); ?>
        </div>
        <div class="mt-4">
            <span class="px-3 py-1 bg-white/20 rounded-full text-sm">
                <?php echo ucfirst($transaction['status']); ?>
            </span>
        </div>
    </div>
    
    <!-- Transaction Details -->
    <div class="space-y-4">
        <div class="bg-gray-50 rounded-lg p-4">
            <h4 class="font-bold text-gray-800 mb-3">
                <i class="fas fa-info-circle mr-2"></i> Transaction Details
            </h4>
            <div class="space-y-2">
                <div class="flex justify-between">
                    <span class="text-gray-600">Transaction ID:</span>
                    <span class="font-medium">#<?php echo str_pad($transaction['id'], 6, '0', STR_PAD_LEFT); ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Type:</span>
                    <span class="font-medium"><?php echo $type_config['label']; ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Status:</span>
                    <span class="font-medium text-<?php echo $status_color; ?>-600">
                        <?php echo ucfirst($transaction['status']); ?>
                    </span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Date:</span>
                    <span class="font-medium"><?php echo date('F d, Y h:i A', strtotime($transaction['created_at'])); ?></span>
                </div>
            </div>
        </div>
        
        <!-- Description -->
        <div class="bg-gray-50 rounded-lg p-4">
            <h4 class="font-bold text-gray-800 mb-3">
                <i class="fas fa-file-alt mr-2"></i> Description
            </h4>
            <p class="text-gray-600"><?php echo htmlspecialchars($transaction['description']); ?></p>
        </div>
        
        <!-- Status Information -->
        <div class="bg-<?php echo $status_color; ?>-50 border border-<?php echo $status_color; ?>-200 rounded-lg p-4">
            <div class="flex">
                <i class="fas fa-info-circle text-<?php echo $status_color; ?>-500 text-xl mr-3 mt-1"></i>
                <div>
                    <h4 class="font-bold text-<?php echo $status_color; ?>-800 mb-2">
                        Status: <?php echo ucfirst($transaction['status']); ?>
                    </h4>
                    <p class="text-<?php echo $status_color; ?>-700 text-sm">
                        <?php
                        switch($transaction['status']) {
                            case 'completed':
                                echo 'This transaction has been successfully processed and completed.';
                                break;
                            case 'pending':
                                echo 'This transaction is currently being processed. Please wait for admin approval.';
                                break;
                            case 'failed':
                                echo 'This transaction failed to process. Please contact support for assistance.';
                                break;
                            default:
                                echo 'Transaction status information.';
                        }
                        ?>
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Actions -->
    <div class="pt-4 border-t">
        <div class="flex justify-center space-x-3">
            <?php if($transaction['status'] == 'pending' && $transaction['type'] == 'deposit'): ?>
                <button onclick="cancelTransaction(<?php echo $transaction['id']; ?>); closeTransactionModal();" 
                        class="bg-red-600 text-white font-bold py-2 px-6 rounded-lg hover:bg-red-700 transition duration-300">
                    <i class="fas fa-times mr-2"></i> Cancel Transaction
                </button>
            <?php endif; ?>
            
            <button onclick="closeTransactionModal()" 
                    class="bg-gray-200 text-gray-700 font-bold py-2 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                <i class="fas fa-times mr-2"></i> Close
            </button>
        </div>
    </div>
</div>

<script>
function cancelTransaction(transactionId) {
    if(!confirm('Are you sure you want to cancel this transaction?')) {
        return;
    }
    
    fetch('ajax/cancel-transaction.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'transaction_id=' + transactionId
    })
    .then(response => response.json())
    .then(data => {
        if(data.success) {
            showNotification('Transaction cancelled successfully!', 'success');
            setTimeout(() => {
                location.reload();
            }, 1500);
        } else {
            showNotification(data.message || 'Error cancelling transaction', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Network error', 'error');
    });
}

function closeTransactionModal() {
    document.getElementById('transactionModal').classList.add('hidden');
    document.getElementById('transactionModal').classList.remove('flex');
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
    notification.innerHTML = `
        <div class="flex items-center">
            <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
            <span>${message}</span>
        </div>
    `;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.classList.remove('animate__fadeInRight');
        notification.classList.add('animate__fadeOutRight');
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}
</script>